//===============================================================================
// Microsoft patterns & practices Enterprise Library
// Configuration Application Block
//===============================================================================
// Copyright  Microsoft Corporation.  All rights reserved.
// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY
// OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT
// LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
// FITNESS FOR A PARTICULAR PURPOSE.
//===============================================================================

using System;
using System.Reflection;

using Microsoft.Practices.EnterpriseLibrary.Configuration.Factory;
namespace Microsoft.Practices.EnterpriseLibrary.Configuration.Injection
{
    /// <summary>
    /// Represents the base class from which all implementations of collection property parameter attributes must derive.
    /// </summary>
    public abstract class CollectionConverterBaseAttribute : ConverterBaseAttribute
    {
        private readonly Type elementFactoryType;

        public CollectionConverterBaseAttribute()
            : this(null)
        {
        }

        /// <summary>
        /// Initialize a new instance of the <see cref="CollectionPropertyParameterAttribute"/> class with the order, the parameter type, the property name and the type of collection to create.
        /// </summary>
        /// <param name="collectionType">The type of collection to create.</param>
        public CollectionConverterBaseAttribute(Type elementFactoryType)
        {
            this.elementFactoryType = elementFactoryType;
        }

        public Type ElementFactoryType
        {
            get { return elementFactoryType; }
        }

        protected object GetElementFactoryGeneric(Type elementType, Type elementConfigurationType)
        {
            MethodInfo getElementFactory = typeof(CollectionConverterBaseAttribute).GetMethod("GetElementFactory", BindingFlags.Instance | BindingFlags.NonPublic);
            getElementFactory = getElementFactory.MakeGenericMethod(elementType, elementConfigurationType);

            object elementFactory = getElementFactory.Invoke(this, new object[0]);
            return elementFactory;
        }

        /// <summary>
        /// When overriden by a class, gets the <see cref="TransformationStrategy"/> to use to transform the nested element.
        /// </summary>
        /// <returns>The <see cref="TransformationStrategy"/> to use to transform the nested element.</returns>
        protected virtual IObjectFactory<TElementToCreate, TElementConfiguration> GetElementFactory<TElementToCreate, TElementConfiguration>()
        {
            if (ElementFactoryType == null)
            {
                return new NullFactory<TElementToCreate, TElementConfiguration>();
            }

            IObjectFactory<TElementToCreate, TElementConfiguration> elementFactory = (IObjectFactory<TElementToCreate, TElementConfiguration>)Activator.CreateInstance(ElementFactoryType);
            return elementFactory;
        }
    }
}
